﻿<?php

class ZUGFeRD {

    const DATE_FIELD = 0;
    const CURRENCY_FIELD = 1;
    const STRING_FIELD = 2;
    const COUNTRY_FIELD = 3;
    const LANGUAGE_FIELD = 4;
    const UNIT_FIELD = 5;
    const QUANTITY_FIELD = 6;

    const BASIC_TEMPLATE = 1;
    const ITEM_TEMPLATE = 2;

    private $index = 0;                       /* line index from control file */
    private $matches = array ();                    /* info from control file */
    private $xmlComfort = "";                         /* xml for comfort bill */
    private $xmlItems = "";                           /* xml for billed items */
    private $language = "";                     /* language for text mappings */
    private $unitCodeMap = null;             /* unit names to ZUFFeRD mapping */
    private $countryCodeMap = null;       /* country names to ZUFFeRD mapping */
    private $languageCodeMap = null;     /* language names to ZUFFeRD mapping */

    public function __construct ($language, $ctlPath) {
        $this->language = $language;

        if (! file_exists($ctlPath))
            $this->error("$ctlPath not found..");

        $ctl = file_get_contents ($ctlPath);
        $this->xmlComfort = file_get_contents ("forms/comfort.xml");
        $this->xmlItems = file_get_contents ("forms/items.xml");

        if (($this->xmlComfort === false) || ($this->xmlItems === false))
        $this->error ("Can't read required xml templates");

        preg_match_all ("/\[([0-9a-zA-Z]{10})\](.+)\r/", $ctl, $this->matches, PREG_SET_ORDER);
    }

    public static function error ($msg) {
        echo $msg."\n";
        file_put_contents ("zugferd.log", "[" . date ('Y-m-d h:i:s', time ()) . "] " . $msg . "\n",   FILE_APPEND);
        die();
    }

    private function readMapFile ($filename) {
        $result = array ();
        $handle = fopen ($filename, "r");
        if ($handle) {
            while (($line = fgets ($handle)) !== false){
                $parts = explode (":", $line);
                if(count ($parts) == 2)
                    $result[trim ($parts [0])] = trim ($parts [1]);
                }
            fclose($handle);
        } else {
            $this->error("could not open " . $filename);
        }
        return $result;
    }

    private function unitToUnitCode($unit) {
        if($this->unitCodeMap == null)
            $this->unitCodeMap = $this->readMapFile("templates/$this->language/unitCodes.csv");
        return $this->unitCodeMap [$unit];
    }

    private function countryToCountryID($country) {
        if($this->countryCodeMap == null)
            $this->countryCodeMap = $this->readMapFile("templates/$this->language/countryCodes.csv");
      return $this->countryCodeMap [$country];
    }

    private function langToLangID($language) {
        if($this->languageCodeMap == null)
            $this->languageCodeMap = $this->readMapFile("templates/$this->language/languageCodes.csv");
        return $this->languageCodeMap [$language];
      }

    public function formatDate($datestring) {
        $date = date_parse ($datestring);
        $out = $date ["year"];
        $out .= strlen ($date ["month"]) < 2 ? "0".$date ["month"] : $date ["month"];
        $out .= strlen ($date ["day"]) < 2 ? "0".$date ["day"] : $date ["day"];
        return $out;
    }

    public function formatQuantity ($quant) {
        $quant = $this->formatCurrency (trim ($quant));
        $count = strlen ($quant) - strpos ($quant, ".") - 1;
        while ($count++ < 4)
            $quant .= "0";
        return $quant;
    }

    public function formatCurrency ($currency) {
        preg_match ('/([ 0-9\.,]+)/', trim ($currency), $m);
        $m [1] = str_replace ('.', '', $m [1]);           /* remove thousands */
        $m [1] = str_replace (',', '.', $m [1]);    /* convert fraction part  */
        return $m [1];
    }

    public function getIndex () {
        return $this->matches [$this->index] [1];
    }

    public function setIndex ($lineIndex){
        for ($idx = 0; $idx < count ($this->matches); $idx++){
            if ($this->matches[$idx][1] == $lineIndex)
              break;
        }
        if ($idx==count ($this->matches))
          return "";

        $this->index = $idx;
        return trim ($this->matches [$idx] [2]);
    }

    public function getLine ($lineIndex){
        $line = "";

        do {
            for ($idx = 0; $idx < count ($this->matches); $idx++){
                if ($this->matches [$idx] [1] == $lineIndex)
                  break;
            }
            if ($idx == count ($this->matches))
                return "";

            $lineIndex = $this->matches [$idx + 1] [1];
            $lineidx = substr ($this->matches [$idx] [1], 0, 6);
            $line .= $this->matches [$idx] [2];
        }
        while (substr ($lineIndex, 0, 6) == $lineidx);

        return $line;
    }

    public function fetch ($lineIndex, &$target, $regEx, $idx, $fieldType = self::STRING_FIELD){
        $target [$idx] = ($lineIndex == "") ? $this->matches [$this->index++] [2]
                                            : $this->setIndex ($lineIndex);

        preg_match ("/" . $regEx . "/", $target [$idx], $m);

        switch ($fieldType) {
            case self::CURRENCY_FIELD:  $target [$idx] = $this->formatCurrency ($m [1]); break;
            case self::DATE_FIELD:      $target [$idx] = $this->formatDate ($m [1]); break;
            case self::LANGUAGE_FIELD:  $target [$idx] = $this->langToLangID ($m [1]); break;
            case self::COUNTRY_FIELD:   $target [$idx] = $this->countryToCountryID ($m [1]); break;
            case self::UNIT_FIELD:      $target [$idx] = $this->unitToUnitCode ($m [1]); break;
            case self::QUANTITY_FIELD:  $target [$idx] = $this->formatQuantity ($m[1]); break;
            case self::STRING_FIELD:    $target [$idx] = $m [1]; break;
        }
    }

    public function substitute ($tmpl, $vars) {
        $line = ($tmpl == self::ITEM_TEMPLATE) ? $this->xmlItems : $this->xmlComfort;
        foreach ($vars as $key => $value){
            $line = str_replace( "@$key@" , $value , $line);
        }
        return $line;
    }
}
?>